import os
import sys
import logging
import time
import random
from concurrent.futures import ThreadPoolExecutor, as_completed
from datetime import datetime, timedelta

import httpx
import pandas as pd
from httpx import Limits
from bs4 import BeautifulSoup
import jwt
import subprocess
import json as json_module
from urllib.parse import quote
import openpyxl  # ensure PyInstaller bundles Excel engine
from pathlib import Path
import atexit
import platform


# === Константы ===
WORKERS_COUNT = 150
WB_ACTIVITY_DAYS = 15
EXCEL_FILENAME = "client_reports.xlsx"
HTTPX_TIMEOUT = 10
MAX_RETRIES = 8
LIMITS = Limits(max_connections=20, max_keepalive_connections=5)
MAX_RETRY_ERROR = 0

def get_base_dir() -> str:
    """Return directory of executable when frozen, else script directory."""
    if getattr(sys, 'frozen', False):
        # Если exe внутри Reports/, поднимаемся на уровень выше
        exe_dir = os.path.dirname(sys.executable)
        if os.path.basename(exe_dir).lower() == 'reports':
            return os.path.dirname(exe_dir)
        return exe_dir
    return os.path.dirname(os.path.abspath(__file__))

def setup_logging() -> logging.Logger:
    # Создаем логгер
    logger = logging.getLogger(__name__)
    logger.setLevel(logging.INFO)
    
    # Очищаем существующие хендлеры
    for handler in logger.handlers[:]:
        logger.removeHandler(handler)
    
    # Хендлер для консоли
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.INFO)
    console_formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
    console_handler.setFormatter(console_formatter)
    logger.addHandler(console_handler)
    
    # Хендлер для файла
    try:
        log_file_path = os.path.join(get_base_dir(), "local_reports.log")
        file_handler = logging.FileHandler(log_file_path, encoding='utf-8', mode='a')
        file_handler.setLevel(logging.INFO)
        file_formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
        file_handler.setFormatter(file_formatter)
        logger.addHandler(file_handler)
    except Exception as e:
        # Если не удалось создать файл логов, продолжаем без него
        pass
    
    # Устанавливаем уровень для httpx
    logging.getLogger("httpx").setLevel(logging.WARNING)
    
    return logger


logger = setup_logging()


# === Класс для подсчета трафика ===
class TrafficCounter:
    def __init__(self):
        self.sent_bytes = 0
        self.received_bytes = 0
        self.request_count = 0

    def add_request(self, sent_bytes=0, received_bytes=0):
        self.sent_bytes += sent_bytes
        self.received_bytes += received_bytes
        self.request_count += 1

    def get_total_bytes(self):
        return self.sent_bytes + self.received_bytes

    def get_stats(self):
        total_mb = self.get_total_bytes() / (1024 * 1024)
        sent_mb = self.sent_bytes / (1024 * 1024)
        received_mb = self.received_bytes / (1024 * 1024)
        return {
            'total_mb': total_mb,
            'sent_mb': sent_mb,
            'received_mb': received_mb,
            'total_bytes': self.get_total_bytes(),
            'sent_bytes': self.sent_bytes,
            'received_bytes': self.received_bytes,
            'request_count': self.request_count
        }


# Глобальный счетчик трафика
traffic_counter = TrafficCounter()


def estimate_request_size(url, headers=None, data=None, json=None, params=None):
    """Оценка размера исходящего запроса"""
    size = len(url.encode('utf-8'))
    if headers:
        for key, value in headers.items():
            size += len(f"{key}: {value}\r\n".encode('utf-8'))
    if data:
        if isinstance(data, str):
            size += len(data.encode('utf-8'))
        elif isinstance(data, dict):
            size += len(str(data).encode('utf-8'))
        else:
            size += len(str(data).encode('utf-8'))
    if json:
        size += len(json_module.dumps(json).encode('utf-8'))
    if params:
        params_str = '&'.join([f"{k}={v}" for k, v in params.items()])
        size += len(params_str.encode('utf-8'))
    return size


def load_proxies(filename: str) -> list[str]:
    """Читает прокси из файла. Поддерживает форматы:
    - http(s)://user:pass@host:port (оставляем как есть)
    - host:port:user:pass → конвертируем в http://user:pass@host:port
    - host:port → конвертируем в http://host:port
    - "mobile" → возвращает пустой список (работа без прокси)
    """
    proxies: list[str] = []
    try:
        with open(filename, 'r', encoding='utf-8') as file:
            content = file.read().strip().lower()
            # Если файл содержит только слово "mobile", возвращаем пустой список
            if content == "mobile":
                logger.info("Обнаружен режим 'mobile' - работа без прокси")
                return []
            
            # Иначе парсим прокси построчно
            file.seek(0)  # Возвращаемся в начало файла
            for raw in file:
                line = raw.strip()
                if not line or line.startswith('#'):
                    continue
                # Формат: host:port:user:pass[:change_url]
                host_port_user_pass_rest = line.split(':', 3)
                if len(host_port_user_pass_rest) >= 4 and host_port_user_pass_rest[1].isdigit():
                    host = host_port_user_pass_rest[0]
                    port = host_port_user_pass_rest[1]
                    user = host_port_user_pass_rest[2]
                    password_and_rest = host_port_user_pass_rest[3]
                    password = password_and_rest.split(':', 1)[0]
                    user_q = quote(user, safe='')
                    pass_q = quote(password, safe='')
                    proxies.append(f"http://{user_q}:{pass_q}@{host}:{port}")
                    continue
                # Готовая URL-строка прокси
                if line.startswith(('http://', 'https://')) and '@' in line:
                    proxies.append(line)
                    continue
                # Простой формат host:port
                parts = line.split(':')
                if len(parts) == 2 and parts[1].isdigit():
                    host, port = parts
                    proxies.append(f"http://{host}:{port}")
                else:
                    # Последняя попытка: оставить как есть
                    proxies.append(line)
    except FileNotFoundError:
        logger.error(f"Файл {filename} не найден")
    return proxies


# === fetch helpers без прокси ===
def fetch_get_no_proxy(url, token, params=None):
    global MAX_RETRY_ERROR
    headers = {"Authorization": f"Bearer {token}"}
    for attempt in range(MAX_RETRIES):
        try:
            sent_size = estimate_request_size(url, headers, params=params)
            response = httpx.get(url, headers=headers, params=params, timeout=HTTPX_TIMEOUT, verify=False)
            response.raise_for_status()
            received_size = len(response.content)
            traffic_counter.add_request(sent_size, received_size)
            return response.json()
        except httpx.HTTPStatusError as e:
            if e.response.status_code == 404:
                return None
        except Exception as e:
            logger.warning(f"Ошибка при GET-запросе без прокси (try {attempt + 1}/{MAX_RETRIES}): {e}")
            if attempt == MAX_RETRIES - 1:
                MAX_RETRY_ERROR += 1
                return None
    MAX_RETRY_ERROR += 1
    return None


# === fetch helpers с прокси ===
def fetch_get_with_proxy(url, token, proxies=None, params=None):
    if proxies is None:
        proxies = []
    global MAX_RETRY_ERROR
    headers = {"Authorization": f"Bearer {token}"}
    for attempt in range(MAX_RETRIES):
        proxy_url = random.choice(proxies) if proxies else None
        try:
            sent_size = estimate_request_size(url, headers, params=params)
            with httpx.Client(
                proxy=proxy_url,
                timeout=HTTPX_TIMEOUT,
                limits=LIMITS,
                verify=False,
            ) as client:
                response = client.get(url, headers=headers, params=params)
                response.raise_for_status()
                received_size = len(response.content)
                traffic_counter.add_request(sent_size, received_size)
                return response.json()
        except httpx.HTTPStatusError as e:
            if e.response.status_code == 404:
                return None
        except Exception as e:
            logger.warning(f"Ошибка при GET-запросе (try {attempt + 1}/{MAX_RETRIES}): {e} | proxy: {proxy_url}")
            if attempt == MAX_RETRIES - 1:
                MAX_RETRY_ERROR += 1
                return None
            time.sleep(0.5 * (attempt + 1))
    MAX_RETRY_ERROR += 1
    return None


def fetch_with_proxy(url, token, proxies=None, json=None, data=None, extra_headers=None):
    if proxies is None:
        proxies = []
    global MAX_RETRY_ERROR
    headers = {"Authorization": f"Bearer {token}"}
    if extra_headers:
        headers.update(extra_headers)
    for attempt in range(MAX_RETRIES):
        proxy_url = random.choice(proxies) if proxies else None
        try:
            sent_size = estimate_request_size(url, headers, data=data, json=json)
            with httpx.Client(
                proxy=proxy_url,
                timeout=HTTPX_TIMEOUT,
                limits=LIMITS,
                verify=False,
            ) as client:
                response = client.post(url, headers=headers, json=json, data=data)
                response.raise_for_status()
                received_size = len(response.content)
                traffic_counter.add_request(sent_size, received_size)
                return response.json()
        except httpx.HTTPStatusError as e:
            if e.response.status_code == 404:
                return None
        except Exception as e:
            logger.warning(
                f"Ошибка при POST-запросе (try {attempt + 1}/{MAX_RETRIES}): {type(e).__name__}: {e} | proxy {proxy_url} "
            )
            if attempt == MAX_RETRIES - 1:
                MAX_RETRY_ERROR += 1
                return None
            time.sleep(0.5 * (attempt + 1))
    MAX_RETRY_ERROR += 1
    return None


def fetch_patch_no_proxy(url, token, params=None):
    global MAX_RETRY_ERROR
    headers = {"Authorization": f"Bearer {token}"}
    for attempt in range(MAX_RETRIES):
        try:
            sent_size = estimate_request_size(url, headers, params=params)
            response = httpx.patch(url, headers=headers, params=params, timeout=HTTPX_TIMEOUT, verify=False)
            response.raise_for_status()
            received_size = len(response.content)
            traffic_counter.add_request(sent_size, received_size)
            return True
        except httpx.HTTPStatusError as e:
            if e.response.status_code == 404:
                return False
            logger.warning(f"HTTP ошибка при PATCH-запросе (try {attempt + 1}/{MAX_RETRIES}): {e.response.status_code}")
        except Exception as e:
            logger.warning(f"Ошибка при PATCH-запросе без прокси (try {attempt + 1}/{MAX_RETRIES}): {e}")
            if attempt == MAX_RETRIES - 1:
                MAX_RETRY_ERROR += 1
                return False
    MAX_RETRY_ERROR += 1
    return False


# === Вспомогательные функции WB API (без изменений логики) ===
def get_shard_key_from_token(token: str) -> str | None:
    try:
        decoded_token = jwt.decode(token, options={"verify_signature": False})
        return decoded_token.get("shard_key")
    except Exception as e:
        logger.error(f"Ошибка декодирования JWT токена: {e}")
        return None


def get_delivery_status_for_item(token: str, proxies=None, uid: str = "") -> str:
    if proxies is None:
        proxies = []
    shard_key = get_shard_key_from_token(token)
    if not shard_key:
        logger.error("Не удалось получить shard_key из токена")
        return ""
    url = f"https://wbx-status-tracker.wildberries.ru/api/v2/last-status?shard={shard_key}"
    request_data = {"ids": [uid]}
    result = fetch_with_proxy(url, token, proxies, json=request_data)
    if result is None:
        return ""
    if isinstance(result, list) and len(result) > 0:
        item = result[0]
        if isinstance(item, dict):
            return item.get("status_name", "")
    return ""


def get_wb_data(token, proxies=None):
    if proxies is None:
        proxies = []
    url = "https://www.wildberries.by/webapi/lk/myorders/delivery/code2"
    result = fetch_with_proxy(url, token, proxies, data={})
    return result if result else None


def get_personal_info(token: str, proxies=None) -> dict:
    """Получить персональные данные через WB /personalinfo.
    Возвращает dict: { 'phone': str, 'fullName': str, 'genderE': str }
    """
    if proxies is None:
        proxies = []
    url = "https://www.wildberries.by/webapi/personalinfo"
    resp = fetch_with_proxy(url, token, proxies, data={})
    result = {"phone": "", "fullName": "", "genderE": ""}
    if not resp or not isinstance(resp, dict):
        return result
    value = resp.get("value", {}) if isinstance(resp, dict) else {}
    # phone из contacts, затем fallback value.phone
    contacts = value.get("contacts", [])
    if isinstance(contacts, list):
        for c in contacts:
            if isinstance(c, dict) and c.get("contact_type") == "phone" and c.get("contact"):
                result["phone"] = str(c.get("contact"))
                break
    if not result["phone"] and value.get("phone") is not None:
        result["phone"] = str(value.get("phone"))
    # fullName
    full_name = value.get("fullName")
    if isinstance(full_name, str):
        result["fullName"] = full_name.strip()
    # genderE
    gender_e = value.get("genderE")
    if isinstance(gender_e, str):
        result["genderE"] = gender_e
    return result


def get_wb_balance(token: str, proxies: list[str]) -> int | float | None:
    if proxies is None:
        proxies = []
    url = "https://www.wildberries.by/webapi/account/getsignedbalance"
    result = fetch_with_proxy(url, token, proxies)
    if result and isinstance(result, dict):
        value = result.get("value", {})
        return value.get("moneyBalanceRUB")
    return None


def get_archive_data(token, fio, phone, proxies=None):
    if proxies is None:
        proxies = []
    url = "https://www.wildberries.by/webapi/lk/myorders/archive/get"
    payload = {"limit": 150, "type": "all", "status": 544}
    result = fetch_with_proxy(url, token, proxies, json=payload)
    if result is None:
        return None
    archive = result.get("value", {}).get("archive", [])
    for item in archive:
        if isinstance(item, dict):
            item["fio"] = fio
            item["phone"] = phone
            if "office" in item and isinstance(item["office"], dict):
                item["office"] = item["office"].get("address", "")
    return archive if archive else None


def get_notevaluated_data(token, fio, phone, proxies=None):
    if proxies is None:
        proxies = []
    url = "https://www.wildberries.by/webapi/lk/myorders/goods/notevaluated?limit=150"
    result = fetch_with_proxy(url, token, proxies)
    if result is None:
        return None
    goods = result.get("value", [])
    for item in goods:
        if isinstance(item, dict):
            item["fio"] = fio
            item["phone"] = phone
            if "office" in item and isinstance(item["office"], dict):
                item["office"] = item["office"].get("address", "")
    return goods if goods else None


def get_feedback_data(token, fio, phone, proxies=None):
    if proxies is None:
        proxies = []
    url = "https://www.wildberries.by/webapi/lk/discussion/feedback/loaddata?page=1&pageSize=100&search="
    result = fetch_with_proxy(url, token, proxies)
    if result is None:
        return None
    feedbacks = []
    raw_items = result.get("value", {}).get("tableItems", [])
    for item in raw_items:
        entity = item.get("entity", {})
        product = item.get("product", {})
        review_data = {**entity, **{
            "product_name": product.get("name"),
            "product_brand": product.get("brand"),
            "product_cod": product.get("cod"),
            "product_link": product.get("link"),
        }}
        review_data["fio"] = fio
        review_data["phone"] = phone
        feedbacks.append(review_data)
    return feedbacks if feedbacks else None


def get_delivery_active_data(token, fio, phone, proxies=None):
    if proxies is None:
        proxies = []
    url = "https://www.wildberries.by/webapi/v2/lk/myorders/delivery/active"
    result = fetch_with_proxy(url, token, proxies)
    if result is None:
        return None
    delivery_active = []
    raw_items = result.get("value", {}).get("positions", [])
    for position in raw_items:
        parsed_order = {**position}
        parsed_order["fio"] = fio
        parsed_order["phone"] = phone
        delivery_active.append(parsed_order)
    return delivery_active if delivery_active else None


def get_delivery_orders_data(token, fio, phone, proxies=None):
    if proxies is None:
        proxies = []
    url = "https://wbxoofex.wildberries.ru/api/v2/orders"
    result = fetch_get_with_proxy(url, token, proxies, params={"limit": 150})
    if result is None:
        return None
    delivery_orders = []
    raw_items = result.get("data", {})
    for order in raw_items:
        order_data = order.copy()
        rids = order_data.pop("rids", [])
        for rid in rids:
            merged = {**order_data, **rid}
            if "nm_id" in merged:
                merged["code1S"] = merged.pop("nm_id")
            uid = merged.get("uid", "")
            if uid:
                delivery_status_api = get_delivery_status_for_item(token, proxies, uid)
                merged["delivery_status_api"] = delivery_status_api
            else:
                merged["delivery_status_api"] = ""
            merged["fio"] = fio
            merged["phone"] = phone
            delivery_orders.append(merged)
    return delivery_orders if delivery_orders else None


def fetch_html(url, params=None):
    global MAX_RETRY_ERROR
    for attempt in range(MAX_RETRIES):
        try:
            sent_size = estimate_request_size(url, params=params)
            response = httpx.get(url, params=params, timeout=HTTPX_TIMEOUT, verify=False)
            response.raise_for_status()
            received_size = len(response.content)
            traffic_counter.add_request(sent_size, received_size)
            return response.text
        except httpx.HTTPStatusError as e:
            if e.response.status_code == 404:
                return None
        except Exception as e:
            logger.warning(f"Ошибка при GET HTML-запросе (try {attempt + 1}/{MAX_RETRIES}): {e}")
            if attempt == MAX_RETRIES - 1:
                MAX_RETRY_ERROR += 1
                return None
            time.sleep(0.5 * (attempt + 1))
    MAX_RETRY_ERROR += 1
    return None


def generate_qrcode_url(qr_str: str) -> str:
    """Генерирует QR-код через API mprating и возвращает ссылку"""
    if not qr_str or qr_str.strip() == "":
        return ""
    from urllib.parse import quote
    encoded_qr = quote(qr_str, safe='')
    return f"https://qr.mprating.ru/qr?text={encoded_qr}"


def parse_receipt_html(receipt_url):
    try:
        html_content = fetch_html(receipt_url)
        if html_content is None:
            logger.warning(f'Не удалось получить HTML чека {receipt_url}')
            return []
        soup = BeautifulSoup(html_content, 'html.parser')
        inn_items = []
        product_rows = soup.find_all('div', class_='products-item')
        total_positions = len(product_rows)
        for row in product_rows:
            inn_div = row.find('div', class_='products-supplier-inn gray')
            if not inn_div:
                inn_div = row.find('div', class_='products-supplier-inn')
            name_div = row.find('div', class_='products-prop-value')
            rid_div = row.find('div', class_='products-prop-value gray')
            cost_div = row.find('div', class_='products-cell products-cell_cost_additional')
            count_div = row.find('div', class_='products-cell products-cell_count')

            name = None
            inn = None
            rid = None
            cost = None
            count = None

            if name_div:
                name = ' '.join(name_div.text.strip().split()[:-1])
            if inn_div:
                inn = inn_div.text.strip().replace('ИНН продавца', '').strip()
                if not inn.isdigit() or len(inn) not in [10, 12]:
                    inn = None
            if rid_div:
                rid = rid_div.text.strip()
            if cost_div:
                val = cost_div.text.strip().replace('\xa0', '').replace('₽', '').replace(' ', '').replace(',', '.')
                try:
                    cost = float(val)
                except Exception:
                    cost = val
            if count_div:
                val = count_div.text.strip()
                try:
                    count = int(val)
                except Exception:
                    count = val

            if name:
                inn_items.append((inn, name, rid, cost, count, total_positions))

        return inn_items
    except Exception as e:
        logger.warning(f'Ошибка парсинга чека {receipt_url}: {e}')
        return []

def read_tokens_from_profiles(profiles_dir: str):
    """Собираем токены из всех подпапок в profiles. Имя папки используем как phone-идентификатор.

    Возвращаем список словарей вида:
    {"token": str, "fio": "", "phone": <folder_name>, "gender": ""}
    """
    tokens = []
    if not os.path.isdir(profiles_dir):
        logger.error(f"Папка profiles не найдена: {profiles_dir}")
        return tokens

    for entry in os.listdir(profiles_dir):
        subdir = os.path.join(profiles_dir, entry)
        if not os.path.isdir(subdir):
            continue
        token_path = os.path.join(subdir, 'token.txt')
        if not os.path.isfile(token_path):
            logger.warning(f"Пропущена папка без token.txt: {subdir}")
            continue
        try:
            with open(token_path, 'r', encoding='utf-8') as f:
                token = f.read().strip()
                if token:
                    tokens.append({
                        "token": token,
                        "fio": "",
                        "phone": entry,  # псевдо телефон = имя папки
                        "gender": "",
                    })
        except Exception as e:
            logger.warning(f"Не удалось прочитать токен из {token_path}: {e}")
    return tokens


def safe_serialize_value(x):
    if isinstance(x, (dict, list)):
        return json_module.dumps(x, ensure_ascii=False)
    return x


def format_datetime(dt_str: str, out_fmt: str = '%Y-%m-%d %H:%M') -> str:
    if not dt_str:
        return ""
    try:
        dt = pd.to_datetime(dt_str, errors='coerce')
        if pd.isna(dt):
            return ""
        return dt.strftime(out_fmt)
    except Exception:
        return ""


def build_excel(proxies: list[str], tokens_data: list[dict], progress_cb=None):
    """Собираем данные по токенам и сохраняем в Excel со вкладками
    'Аккаунты', 'Покупки', 'Доставки', 'Чеки'.

    ВНИМАНИЕ: Все сетевые запросы выполняются существующими функциями из browser_reports_to_back
    без изменений их логики.
    """

    # --- Определяем реальный телефон из personalinfo и сохраняем имя профиля ---
    def get_phone_wrapper(item):
        return item, get_personal_info(item["token"], proxies)

    if tokens_data:
        total_profiles = len(tokens_data)
        processed_profiles = 0
        if callable(progress_cb):
            try:
                progress_cb(processed_profiles, total_profiles)
            except Exception:
                pass
        with ThreadPoolExecutor(max_workers=min(WORKERS_COUNT, len(tokens_data))) as executor:
            futures = [executor.submit(get_phone_wrapper, item) for item in tokens_data]
            for future in as_completed(futures):
                item, pinfo = future.result()
                profile_name = item.get("phone", "")  # старое поле phone содержит имя папки
                item["profile"] = profile_name
                # phone
                if pinfo.get("phone"):
                    item["phone"] = pinfo["phone"]
                # fio (Имя)
                if pinfo.get("fullName"):
                    item["fio"] = pinfo["fullName"]
                # gender from genderE
                gender_e = pinfo.get("genderE", "")
                if isinstance(gender_e, str) and gender_e:
                    ge = gender_e.strip().lower()
                    if ge in ("male", "female"):
                        item["gender"] = ge
                    else:
                        item["gender"] = gender_e
                processed_profiles += 1
                if callable(progress_cb):
                    try:
                        progress_cb(processed_profiles, total_profiles)
                    except Exception:
                        pass

    # --- Балансы ---
    balances: dict[str, float | int | None] = {}

    def get_balance_wrapper(item):
        return item["token"], get_wb_balance(item["token"], proxies)

    if tokens_data:
        with ThreadPoolExecutor(max_workers=min(WORKERS_COUNT, len(tokens_data))) as executor:
            futures = [executor.submit(get_balance_wrapper, item) for item in tokens_data]
            for future in as_completed(futures):
                token, balance = future.result()
                balances[token] = balance

    # --- QR/Коды забора ---
    wb_data = []  # для листа Аккаунты и индексов

    def get_wb_code_wrapper(item):
        return item, get_wb_data(item["token"], proxies)

    wb_results = []
    if tokens_data:
        with ThreadPoolExecutor(max_workers=min(WORKERS_COUNT, len(tokens_data))) as executor:
            futures = [executor.submit(get_wb_code_wrapper, item) for item in tokens_data]
            for future in as_completed(futures):
                item, result = future.result()
                wb_results.append((item, result))

    for item, result in wb_results:
        token = item.get("token")
        phone = item.get("phone")
        if result and isinstance(result, dict) and "value" in result:
            value = result.get("value")
            expire_time = datetime.now() + timedelta(minutes=value.get("expireM", 0))
            private_code = value.get("privateCode")
            qr_str = value.get("qrStr")
            qr_expire = expire_time.strftime("%Y-%m-%d")
        else:
            private_code = ""
            qr_str = ""
            qr_expire = ""

        wb_data.append({
            "fio": item.get("fio", ""),
            "phone": phone,
            "profile": item.get("profile", ""),
            "gender": item.get("gender", ""),
            "privateCode": private_code,
            "qrStr": qr_str,
            "qr_expire_data": qr_expire,
            "balance": balances.get(token),
            "token": token,
        })

    logger.info(f"Данные для листа 'Аккаунты': {len(wb_data)} записей")

    # --- Индексы для дальнейших шагов ---
    wb_index_by_phone = {item['phone']: item for item in wb_data}
    fio_index = {item['phone']: item.get('fio', '') for item in wb_data}
    gender_index = {item['phone']: item.get('gender', '') for item in wb_data}

    # --- Загружаем данные для отчетов (из WB API через существующие функции) ---
    def archive_wrapper(item):
        return get_archive_data(item["token"], item["fio"], item["phone"], proxies)

    def notevaluated_wrapper(item):
        return get_notevaluated_data(item["token"], item["fio"], item["phone"], proxies)

    def feedback_wrapper(item):
        return get_feedback_data(item["token"], item["fio"], item["phone"], proxies)

    def delivery_active_wrapper(item):
        return get_delivery_active_data(item["token"], item["fio"], item["phone"], proxies)

    def delivery_orders_wrapper(item):
        return get_delivery_orders_data(item["token"], item["fio"], item["phone"], proxies)

    archive_data = []
    notevaluated_data = []
    feedback_data = []
    delivery_active_data = []
    delivery_orders_data = []

    if tokens_data:
        with ThreadPoolExecutor(max_workers=WORKERS_COUNT) as executor:
            futures_a = [executor.submit(archive_wrapper, item) for item in tokens_data]
            futures_n = [executor.submit(notevaluated_wrapper, item) for item in tokens_data]
            futures_f = [executor.submit(feedback_wrapper, item) for item in tokens_data]
            futures_da = [executor.submit(delivery_active_wrapper, item) for item in tokens_data]
            futures_do = [executor.submit(delivery_orders_wrapper, item) for item in tokens_data]

            for f in as_completed(futures_a):
                data = f.result()
                if data:
                    archive_data.extend(data)
            for f in as_completed(futures_n):
                data = f.result()
                if data:
                    notevaluated_data.extend(data)
            for f in as_completed(futures_f):
                data = f.result()
                if data:
                    feedback_data.extend(data)
            for f in as_completed(futures_da):
                data = f.result()
                if data:
                    delivery_active_data.extend(data)
            for f in as_completed(futures_do):
                data = f.result()
                if data:
                    delivery_orders_data.extend(data)

    # --- Индексы для статусов (как в browser_reports_to_back) ---
    notevaluated_index = {(item['phone'], item.get('code1S')): item for item in notevaluated_data if isinstance(item, dict)}
    feedback_index = {(item['phone'], item.get('product_cod')): item for item in feedback_data if isinstance(item, dict)}

    # --- Преобразование статусов покупки ---
    purchase_status_translate = {
        'ExcludedFromRate': 'Исключено из рейтинга',
        'FailedPayment': 'Ошибка оплаты',
        'IssuedAwaitingPayment': 'Ожидает оплаты',
        'Purchased': 'Выкуплен',
        'Refund': 'Возврат',
        'Rejected': 'Отклонён',
        'TransitRefund': 'Возврат в пути',
    }

    def make_summary_row(item: dict, field_map: dict, extra_fields: dict):
        row = {}
        for summary_field, item_field in field_map.items():
            if item_field is None:
                row[summary_field] = ""
            else:
                row[summary_field] = item.get(item_field, "")
        row.update(extra_fields)
        return row

    # --- Карты полей, как в browser_reports_to_back ---
    archive_map = {
        "fio": None, "gender": None, "name": "name", "code1S": "code1S", "brand": "brand", "address": "office",
        "phone": "phone", "delivery_status": None, "purchase_status": None, "status": None, "privateCode": None,
        "qrStr": None, "qrcoder": None, "qrserver": None, "review_text": None, "review_success": None,
        "review_date": None,
        "review_photo": None, "review_video": None, "orderDate": "orderDate", "lastDate": "lastDate",
        "price": "price"
    }
    active_map = {
        "fio": None, "gender": None, "name": "name", "code1S": "code1S", "brand": "brand", "address": "address",
        "phone": "phone", "delivery_status": "trackingStatus", "purchase_status": None, "status": None,
        "privateCode": None,
        "qrStr": None, "qrcoder": None, "qrserver": None, "mprating": None, "review_text": None, "review_success": None,
        "review_date": None,
        "review_photo": None, "review_video": None, "orderDate": "orderDate", "lastDate": None,
        "price": "price"
    }
    orders_map = {
        "fio": None, "gender": None, "name": "name", "code1S": "code1S", "brand": "brand",
        "address": "full_address",
        "phone": "phone", "delivery_status": None, "purchase_status": None, "status": None, "privateCode": None,
        "qrStr": None, "qrcoder": None, "qrserver": None, "mprating": None, "review_text": None, "review_success": None,
        "review_date": None,
        "review_photo": None, "review_video": None, "orderDate": None, "lastDate": None, "price": "total_price"
    }

    # --- Итоги ---
    summary_purchases = []
    summary_deliveries = []


    # Покупки (archive)
    for item in archive_data:
        if not isinstance(item, dict):
            continue
        order_date_raw = item.get('orderDate', '')
        try:
            order_date_dt = pd.to_datetime(order_date_raw, errors='coerce')
        except Exception:
            order_date_dt = None
        if order_date_dt is pd.NaT or order_date_dt is None:
            continue
        phone = item.get('phone')
        code1S = item.get('code1S')
        purchase_status_raw = item.get('status', "")
        purchase_status = purchase_status_translate.get(purchase_status_raw, purchase_status_raw)
        status = ""
        if (phone, code1S) in notevaluated_index:
            status = "Ждет отзыва"
        elif (phone, code1S) in feedback_index:
            status = "Отзыв оставлен"
        if purchase_status in ("Отклонён", "Отменен магазином", "Ошибка оплаты", "Исключено из рейтинга", "Возврат"):
            status = purchase_status
        privateCode = wb_index_by_phone.get(phone, {}).get('privateCode', "")
        qrStr = wb_index_by_phone.get(phone, {}).get('qrStr', "")
        qrcoder = f"https://qrcoder.ru/code/?{qrStr}&8&0" if qrStr else ""
        qrserver = f"https://api.qrserver.com/v1/create-qr-code/?size=300x300&data={qrStr}" if qrStr else ""
        mprating = generate_qrcode_url(qrStr) if qrStr else ""
        fio = fio_index.get(phone, "")
        gender = gender_index.get(phone, "")
        review_text = ""
        review_success = ""
        review_date = ""
        review_photo = ""
        review_video = ""
        if (phone, code1S) in feedback_index:
            fb = feedback_index[(phone, code1S)]
            # формат текста как в исходнике (функция format_review_text недоступна здесь без импорта),
            # поэтому оставляем исходные поля фото/видео/дата/успех
            text = fb.get('text', '').strip()
            pros = fb.get('pros', '').strip()
            cons = fb.get('cons', '').strip()
            fields_count = sum([bool(text), bool(pros), bool(cons)])
            if (fields_count >= 2) or (fields_count == 1 and cons):
                parts = []
                if text:
                    parts.append(f"Текст:\n{text}")
                if cons:
                    parts.append(f"Недостатки:\n{cons}")
                if pros:
                    parts.append(f"Достоинства:\n{pros}")
                review_text = '\n\n'.join(parts)
            else:
                review_text = text or pros or ""
            review_success_bool = not fb.get('excludedFromRating', False)
            review_success = "Прошел" if review_success_bool else "Не прошел"
            review_date_dt = pd.to_datetime(fb.get('postDate', ''), errors='coerce')
            review_date = review_date_dt.strftime('%Y-%m-%d') if review_date_dt is not pd.NaT else ""
            review_photo = "Есть" if fb.get('photo') else ""
            review_video = "Есть" if fb.get('video') else ""
        extra = {
            "fio": fio, "gender": gender, "delivery_status": "", "purchase_status": purchase_status,
            "status": status,
            "privateCode": privateCode, "qrStr": qrStr, "qrcoder": qrcoder, "qrserver": qrserver,
            "review_text": review_text, "review_success": review_success, "review_date": review_date,
            "review_photo": review_photo, "review_video": review_video,
        }
        summary_purchases.append(make_summary_row(item, archive_map, extra))

    # Доставки (active)
    # Получаем адреса ПВЗ по officeId для active доставок
    token_to_office_ids_active = {}
    for item in delivery_active_data:
        if not isinstance(item, dict):
            continue
        if not item.get('address') and item.get('officeId'):
            phone = item.get('phone')
            # Находим токен для этого телефона
            wb_token = wb_index_by_phone.get(phone, {}).get('token')
            if wb_token:
                office_id = str(item['officeId'])
                if wb_token not in token_to_office_ids_active:
                    token_to_office_ids_active[wb_token] = {'ids': set(), 'items': []}
                token_to_office_ids_active[wb_token]['ids'].add(office_id)
                token_to_office_ids_active[wb_token]['items'].append(item)
    
    # Для каждого токена делаем запрос с его office_id
    for wb_token, data in token_to_office_ids_active.items():
        office_ids = data['ids']
        items = data['items']
        
        if office_ids:
            ids_params = '&'.join([f'ids={oid}' for oid in office_ids])
            url = 'https://www.wildberries.by/webapi/lk/myorders/delivery/offices'
            
            resp = fetch_with_proxy(
                url, wb_token, proxies, data=ids_params,
                extra_headers={"Content-Type": "application/x-www-form-urlencoded; charset=UTF-8"}
            )
            
            if resp and 'value' in resp:
                offices = resp['value']
                for item in items:
                    office_id = str(item.get('officeId', ''))
                    address = offices.get(office_id, {}).get('address')
                    if address:
                        item['address'] = address
            else:
                logger.warning(f'Ошибка получения адресов ПВЗ для active доставок, токен {wb_token[:10]}...: {resp}')
    
    # Обрабатываем доставки (active) с заполненными адресами
    for item in delivery_active_data:
        if not isinstance(item, dict):
            continue
        order_date_raw = item.get('orderDate', '')
        try:
            order_date_dt = pd.to_datetime(order_date_raw, errors='coerce')
        except Exception:
            order_date_dt = None
        if order_date_dt is pd.NaT or order_date_dt is None:
            continue
        phone = item.get('phone')
        delivery_status = item.get('trackingStatus', "")
        prepaid = item.get('prepaid', None)
        if prepaid == 0:
            status = "Не оплачен"
        elif delivery_status == "Готов к выдаче":
            status = "Можно забрать"
        else:
            status = "В доставке"
        privateCode = wb_index_by_phone.get(phone, {}).get('privateCode', "")
        qrStr = wb_index_by_phone.get(phone, {}).get('qrStr', "")
        qrcoder = f"https://qrcoder.ru/code/?{qrStr}&8&0" if qrStr else ""
        qrserver = f"https://api.qrserver.com/v1/create-qr-code/?size=300x300&data={qrStr}" if qrStr else ""
        mprating = generate_qrcode_url(qrStr) if qrStr else ""
        fio = fio_index.get(phone, "")
        gender = gender_index.get(phone, "")
        
        extra = {
            "fio": fio, "gender": gender, "purchase_status": "", "status": status,
            "privateCode": privateCode, "qrStr": qrStr, "qrcoder": qrcoder, "qrserver": qrserver, "mprating": mprating,
            "review_text": "", "review_success": "", "review_date": "", "review_photo": "", "review_video": "",
            "delivery_status": delivery_status,
        }
        summary_deliveries.append(make_summary_row(item, active_map, extra))

    # Доставки (orders)
    # Получаем адреса ПВЗ по dst_office_id
    token_to_office_ids = {}
    for item in delivery_orders_data:
        if not isinstance(item, dict):
            continue
        if not item.get('full_address') and item.get('dst_office_id'):
            phone = item.get('phone')
            # Находим токен для этого телефона
            wb_token = wb_index_by_phone.get(phone, {}).get('token')
            if wb_token:
                office_id = str(item['dst_office_id'])
                if wb_token not in token_to_office_ids:
                    token_to_office_ids[wb_token] = {'ids': set(), 'items': []}
                token_to_office_ids[wb_token]['ids'].add(office_id)
                token_to_office_ids[wb_token]['items'].append(item)
    
    # Для каждого токена делаем запрос с его office_id
    for wb_token, data in token_to_office_ids.items():
        office_ids = data['ids']
        items = data['items']
        
        if office_ids:
            ids_params = '&'.join([f'ids={oid}' for oid in office_ids])
            url = 'https://www.wildberries.by/webapi/lk/myorders/delivery/offices'
            
            resp = fetch_with_proxy(
                url, wb_token, proxies, data=ids_params,
                extra_headers={"Content-Type": "application/x-www-form-urlencoded; charset=UTF-8"}
            )
            
            if resp and 'value' in resp:
                offices = resp['value']
                for item in items:
                    office_id = str(item.get('dst_office_id', ''))
                    address = offices.get(office_id, {}).get('address')
                    if address:
                        item['full_address'] = address
            else:
                logger.warning(f'Ошибка получения адресов ПВЗ для токена {wb_token[:10]}...: {resp}')
    
    # Обрабатываем доставки (orders) с заполненными адресами
    for item in delivery_orders_data:
        if not isinstance(item, dict):
            continue
        phone = item.get('phone')
        privateCode = wb_index_by_phone.get(phone, {}).get('privateCode', "")
        qrStr = wb_index_by_phone.get(phone, {}).get('qrStr', "")
        qrcoder = f"https://qrcoder.ru/code/?{qrStr}&8&0" if qrStr else ""
        qrserver = f"https://api.qrserver.com/v1/create-qr-code/?size=300x300&data={qrStr}" if qrStr else ""
        mprating = generate_qrcode_url(qrStr) if qrStr else ""
        fio = fio_index.get(phone, "")
        gender = gender_index.get(phone, "")
        price = item.get('total_price', "")
        if isinstance(price, (int, float)):
            price = round(price / 100, 2)
        elif isinstance(price, str) and price.isdigit():
            price = round(int(price) / 100, 2)
        delivery_status = item.get('delivery_status_api', 'Ручная проверка')
        if "Готов" in delivery_status or "Доставлен в постамат" in delivery_status:
            status = "Можно забрать"
        elif delivery_status == "":
            status = "Ручная проверка"
        else:
            status = "В доставке"
        order_date = ""
        if 'order_dt' in item and item['order_dt']:
            try:
                order_date_dt = pd.to_datetime(int(item['order_dt']), errors='coerce', unit='s', utc=True)
                order_date = order_date_dt.strftime('%Y-%m-%d') if order_date_dt is not pd.NaT else ""
            except Exception:
                order_date = ""
        extra = {
            "fio": fio, "gender": gender, "delivery_status": delivery_status, "purchase_status": "",
            "status": status,
            "privateCode": privateCode, "qrStr": qrStr, "qrcoder": qrcoder, "qrserver": qrserver, "mprating": mprating,
            "review_text": "", "review_success": "", "review_date": "", "review_photo": "", "review_video": "",
            "orderDate": order_date, "lastDate": "", "price": price,
        }
        summary_deliveries.append(make_summary_row(item, orders_map, extra))

    # Порядок столбцов как в backend_reports_to_gapi
    purchases_columns_order = [
        "fio", "gender", "phone", "name", "code1S", "brand", "address",
        "purchase_status", "status",
        "review_text", "review_success", "review_date", "review_photo", "review_video",
        "orderDate", "lastDate", "price"
    ]
    deliveries_columns_order = [
        "fio", "gender", "phone", "name", "code1S", "brand", "address",
        "delivery_status", "status", "privateCode", "qrStr", "qrcoder", "qrserver", "mprating",
        "orderDate", "price"
    ]

    # Создаем DataFrame'ы
    # Аккаунты
    df_accounts = pd.DataFrame(wb_data)
    if 'gender' in df_accounts.columns:
        df_accounts['gender'] = df_accounts['gender'].map({'male': 'Мужской', 'female': 'Женский'}).fillna(df_accounts['gender'])
    if 'token' in df_accounts.columns:
        df_accounts = df_accounts.drop(columns=['token'])
    accounts_columns_map = {
        "profile": "Профиль",
        "fio": "Имя",
        "phone": "Телефон",
        "gender": "Пол",
        "privateCode": "Код получения",
        "qrStr": "QR-код",
        "qr_expire_data": "Время действия QR-кода",
        "balance": "Баланс"
    }
    df_accounts = df_accounts.rename(columns=accounts_columns_map)
    df_accounts = df_accounts.replace([float('inf'), float('-inf')], None).fillna('')
    df_accounts = df_accounts.map(safe_serialize_value)

    # Покупки
    if summary_purchases:
        df_purchases = pd.DataFrame(summary_purchases)
        available_purchases_cols = [col for col in purchases_columns_order if col in df_purchases.columns]
        df_purchases = df_purchases[available_purchases_cols]
    else:
        df_purchases = pd.DataFrame(columns=purchases_columns_order)
    df_purchases = df_purchases.replace([float('inf'), float('-inf')], None).fillna('')
    df_purchases = df_purchases.map(safe_serialize_value)
    # Русификация названий столбцов (как в Google Таблицах)
    summary_columns_map = {
        "fio": "ФИО",
        "gender": "Пол",
        "name": "Название",
        "code1S": "Артикул",
        "brand": "Бренд",
        "address": "Адрес",
        "phone": "Телефон",
        "delivery_status": "Статус доставки",
        "purchase_status": "Статус покупки",
        "status": "Статус",
        "privateCode": "Код забора",
        "qrStr": "QR-код (строка)",
        "qrcoder": "QR-код (1)",
        "qrserver": "QR-код (2)",
        "mprating": "QR-код (3)",
        "review_text": "Текст отзыва",
        "review_success": "Успешность отзыва",
        "review_date": "Дата отзыва",
        "review_photo": "Фото-отзыв",
        "review_video": "Видео-отзыв",
        "orderDate": "Дата выкупа",
        "lastDate": "Дата забора",
        "price": "Стоимость",
    }
    df_purchases = df_purchases.rename(columns=summary_columns_map)

    # Доставки
    if summary_deliveries:
        df_deliveries = pd.DataFrame(summary_deliveries)
        available_deliveries_cols = [col for col in deliveries_columns_order if col in df_deliveries.columns]
        df_deliveries = df_deliveries[available_deliveries_cols]
    else:
        df_deliveries = pd.DataFrame(columns=deliveries_columns_order)
    df_deliveries = df_deliveries.replace([float('inf'), float('-inf')], None).fillna('')
    df_deliveries = df_deliveries.map(safe_serialize_value)
    df_deliveries = df_deliveries.rename(columns=summary_columns_map)

    # Ждут отзыва
    if notevaluated_data:
        df_waiting_reviews = pd.DataFrame(notevaluated_data)
        # Переупорядочиваем столбцы: phone, fio в начало
        waiting_reviews_columns = ["phone", "fio"] + [c for c in df_waiting_reviews.columns if c not in {"phone", "fio"}]
        df_waiting_reviews = df_waiting_reviews[waiting_reviews_columns]
        df_waiting_reviews = df_waiting_reviews.replace([float('inf'), float('-inf')], None).fillna('')
        df_waiting_reviews = df_waiting_reviews.map(safe_serialize_value)
        # Русификация названий столбцов
        waiting_reviews_columns_map = {
            "phone": "Телефон",
            "fio": "ФИО", 
            "name": "Название",
            "code1S": "Артикул",
            "brand": "Бренд",
            "office": "Адрес ПВЗ"
        }
        df_waiting_reviews = df_waiting_reviews.rename(columns=waiting_reviews_columns_map)
    else:
        df_waiting_reviews = pd.DataFrame()

    # Оставленные отзывы  
    if feedback_data:
        df_reviews = pd.DataFrame(feedback_data)
        # Переупорядочиваем столбцы: phone, fio в начало
        reviews_columns = ["phone", "fio"] + [c for c in df_reviews.columns if c not in {"phone", "fio"}]
        df_reviews = df_reviews[reviews_columns]
        df_reviews = df_reviews.replace([float('inf'), float('-inf')], None).fillna('')
        df_reviews = df_reviews.map(safe_serialize_value)
        # Русификация названий столбцов
        reviews_columns_map = {
            "phone": "Телефон",
            "fio": "ФИО",
            "product_name": "Название товара", 
            "product_brand": "Бренд",
            "product_cod": "Артикул",
            "product_link": "Ссылка на товар",
            "text": "Текст отзыва",
            "pros": "Достоинства", 
            "cons": "Недостатки",
            "mark": "Оценка",
            "postDate": "Дата публикации",
            "excludedFromRating": "Исключен из рейтинга"
        }
        df_reviews = df_reviews.rename(columns=reviews_columns_map)
    else:
        df_reviews = pd.DataFrame()

    # Чеки (локально собираем из WB без бэкенда)
    df_receipts = collect_receipts_locally(tokens_data, proxies)

    # Пишем Excel
    excel_path = os.path.join(get_base_dir(), EXCEL_FILENAME)
    with pd.ExcelWriter(excel_path, engine='openpyxl') as writer:
        df_accounts.to_excel(writer, sheet_name='Аккаунты', index=False)
        df_purchases.to_excel(writer, sheet_name='Покупки', index=False)
        df_deliveries.to_excel(writer, sheet_name='Доставки', index=False)
        if not df_waiting_reviews.empty:
            df_waiting_reviews.to_excel(writer, sheet_name='Ждут_отзыва', index=False)
        if not df_reviews.empty:
            df_reviews.to_excel(writer, sheet_name='Оставлены_отзывы', index=False)
        if df_receipts is not None:
            df_receipts.to_excel(writer, sheet_name='Чеки', index=False)

    logger.info(f"Excel сохранен: {excel_path}")
    
    # Создаем/перезаписываем last_date.txt с текущим временем в UNIX формате
    last_date_path = os.path.join(get_base_dir(), "last_date.txt")
    try:
        import time
        unix_timestamp = int(time.time())
        with open(last_date_path, 'w', encoding='utf-8') as f:
            f.write(str(unix_timestamp))
        logger.info(f"last_date.txt сохранен: {unix_timestamp}")
    except Exception as e:
        logger.warning(f"Не удалось создать last_date.txt: {e}")


def collect_receipts_locally(tokens_data: list[dict], proxies: list[str]):
    """Собираем чеки напрямую из WB API по каждому токену и приводим к формату
    как в Google-таблице из backend_reports_to_gapi.get_receipts_for_report
    """

    RECEIPTS_PER_PAGE = 50

    def fetch_receipts(token: str):
        """Итератор по всем чекам для заданного токена (с пагинацией)"""

        next_uid = None
        while True:
            url = 'https://astro.wildberries.ru/api/v1/receipt-api/v1/receipts'
            params = {'receiptsPerPage': RECEIPTS_PER_PAGE}
            if next_uid:
                params['nextReceiptUid'] = next_uid

            data = fetch_get_with_proxy(url, token, proxies, params=params)
            if not data:
                break
            receipts = (
                data.get('data', {})
                .get('result', {})
                .get('data', {})
                .get('receipts', [])
            )
            for receipt in receipts:
                yield receipt

            next_uid = data.get('data', {}).get('result', {}).get('nextReceiptUid', '')
            if not next_uid:
                break

    # Сбор в многопоточке
    all_rows = []

    def process_user_receipts(user: dict):
        token = user['token']
        phone = user['phone']
        user_rows = []
        for receipt in fetch_receipts(token):
            receipt_uid = receipt.get('receiptUid')
            link = receipt.get('link')
            operation_sum = receipt.get('operationSum')
            operation_type_id = receipt.get('operationTypeId')
            operation_type = (
                'Покупка' if operation_type_id == 1 else 'Возврат' if operation_type_id == 2 else str(operation_type_id)
            )
            operation_datetime = receipt.get('operationDateTime')
            formatted_datetime = operation_datetime
            if operation_datetime:
                try:
                    dt = pd.to_datetime(operation_datetime)
                    formatted_datetime = dt.strftime('%Y-%m-%d %H:%M')
                except Exception:
                    pass

            # Парсинг HTML чека (используем исходную функцию)
            inn_items = parse_receipt_html(link) if link else []

            for inn, name, rid, cost, count, total_positions in inn_items:
                user_rows.append({
                    'Телефон WB': phone,
                    'ИНН': inn or '',
                    'rid': rid or '',
                    'Наименование товара': name or '',
                    'Цена позиции': cost if cost is not None else '',
                    'Количество товара': count if count is not None else '',
                    'Всего позиций в чеке': total_positions,
                    'Сумма чека': operation_sum,
                    'Дата операции': formatted_datetime,
                    'Тип операции': operation_type,
                    'Ссылка': link or '',
                })
        return user_rows

    if tokens_data:
        with ThreadPoolExecutor(max_workers=WORKERS_COUNT) as executor:
            futures = [executor.submit(process_user_receipts, user) for user in tokens_data]
            for future in as_completed(futures):
                rows = future.result()
                if rows:
                    all_rows.extend(rows)

    if not all_rows:
        logger.info('Нет данных чеков для отчета.')
        return pd.DataFrame(columns=[
            'Телефон WB','ИНН','rid','Наименование товара','Цена позиции','Количество товара',
            'Всего позиций в чеке','Сумма чека','Дата операции','Тип операции','Ссылка'
        ])

    df_receipts = pd.DataFrame(all_rows)
    df_receipts = df_receipts.replace([float('inf'), float('-inf')], None).fillna('')
    df_receipts = df_receipts.map(safe_serialize_value)
    logger.info(f"Данные чеков для отчета успешно получены: {len(df_receipts)} записей.")
    return df_receipts


def resolve_path(*relative_parts: str) -> str:
    """Resolve a resource path near the executable, with macOS .app fallbacks.

    Checks: base_dir, base_dir/.., base_dir/../.., base_dir/../../..
    Returns the first existing path, or the base_dir path if none exist yet.
    """
    base = Path(get_base_dir())
    candidates = [base, base.parent, base.parent.parent, base.parent.parent.parent]
    for candidate in candidates:
        try:
            path = candidate.joinpath(*relative_parts)
            if path.exists():
                return str(path)
        except Exception:
            continue
    return str(base.joinpath(*relative_parts))


def main():
    base_dir = get_base_dir()
    profiles_dir = resolve_path('profiles')

    tokens_data = read_tokens_from_profiles(profiles_dir)
    if not tokens_data:
        logger.error("Токены не найдены. Убедитесь, что папка 'profiles' содержит подпапки с 'token.txt'.")
        return

    proxies = load_proxies(resolve_path('proxy.txt'))
    build_excel(proxies, tokens_data)


def run_gui():
    """Мини-GUI без консоли: показывает крутилку и завершает с сообщением."""
    import threading
    import tkinter as tk
    from tkinter import ttk, messagebox
    from tkinter import PhotoImage

    base_dir = get_base_dir()

    root = tk.Tk()
    root.title("Отчет WB")
    root.geometry("360x120")
    root.resizable(False, False)

    # Иконка окна/панели задач: macOS -> Icon.icns (через AppKit, если доступен), Windows -> Icon.ico
    try:
        if sys.platform == 'darwin':
            icns_path = resolve_path('Icon.icns')
            if os.path.exists(icns_path):
                try:
                    # Пытаемся через динамический импорт (без жесткой зависимости на pyobjc)
                    import importlib
                    appkit = importlib.import_module('AppKit')
                    NSApplication = getattr(appkit, 'NSApplication', None)
                    NSImage = getattr(appkit, 'NSImage', None)
                    if NSApplication and NSImage:
                        app = NSApplication.sharedApplication()
                        img = NSImage.alloc().initWithContentsOfFile_(icns_path)
                        if img:
                            app.setApplicationIconImage_(img)
                except Exception:
                    # Фолбэк: попробуем PNG (если есть)
                    try:
                        png_fallback = resolve_path('Icon.png')
                        if os.path.exists(png_fallback):
                            icon_img = PhotoImage(file=png_fallback)
                            root.iconphoto(True, icon_img)
                    except Exception:
                        pass
        elif sys.platform.startswith('win'):
            ico_path = resolve_path('Icon.ico')
            if os.path.exists(ico_path):
                try:
                    root.iconbitmap(default=ico_path)
                except Exception:
                    pass
                # Устанавливаем AppUserModelID, чтобы таскбар использовал нашу иконку и корректную группировку
                try:
                    import ctypes
                    ctypes.windll.shell32.SetCurrentProcessExplicitAppUserModelID("mprating.LocalReports")
                except Exception:
                    pass
    except Exception:
        pass

    frame = ttk.Frame(root, padding=12)
    frame.pack(fill=tk.BOTH, expand=True)

    progress_label_var = tk.StringVar()
    progress_label = ttk.Label(frame, textvariable=progress_label_var, anchor="center", justify="center")
    progress_label.pack(pady=(0, 8), fill=tk.X)
    try:
        # Перенос строк по ширине окна
        progress_label.configure(wraplength=300)
    except Exception:
        pass
    progress_label_var.set("Идет формирование отчета...")

    progress = ttk.Progressbar(frame, mode="indeterminate")
    progress.pack(fill=tk.X)
    progress.start(10)

    # Контейнер для кнопок после завершения
    buttons_frame = ttk.Frame(frame)
    buttons_frame.pack(fill=tk.X, pady=(10, 0))
    for child in list(buttons_frame.children.values()):
        child.destroy()

    def worker():
        try:
            profiles_dir = resolve_path('profiles')
            tokens_data = read_tokens_from_profiles(profiles_dir)
            proxies = load_proxies(resolve_path('proxy.txt'))

            def gui_progress_cb(done, total):
                root.after(0, lambda: progress_label_var.set(f"Обработано профилей: {done} из {total}"))

            build_excel(proxies, tokens_data, progress_cb=gui_progress_cb)
            excel_path = os.path.join(base_dir, EXCEL_FILENAME)
            def open_report():
                try:
                    if sys.platform.startswith('win'):
                        os.startfile(excel_path)
                    elif sys.platform == 'darwin':
                        subprocess.call(['open', excel_path])
                    else:
                        subprocess.call(['xdg-open', excel_path])
                except Exception:
                    pass

            def show_folder():
                try:
                    folder = os.path.dirname(excel_path)
                    if sys.platform.startswith('win'):
                        os.startfile(folder)
                    elif sys.platform == 'darwin':
                        subprocess.call(['open', folder])
                    else:
                        subprocess.call(['xdg-open', folder])
                except Exception:
                    pass

            def on_done():
                progress.destroy()
                progress_label_var.set(f"Отчёт сохранён в:\n{excel_path}")
                for child in list(buttons_frame.children.values()):
                    child.destroy()
                btn_open = ttk.Button(buttons_frame, text="Открыть", command=open_report)
                btn_show = ttk.Button(buttons_frame, text="Показать", command=show_folder)
                btn_open.pack(side=tk.LEFT, expand=True, fill=tk.X, padx=(0, 6))
                btn_show.pack(side=tk.LEFT, expand=True, fill=tk.X)

            root.after(0, on_done)
        except Exception as e:
            def show_error():
                progress.stop()
                messagebox.showerror("Ошибка", f"Сбой: {e}")
                root.destroy()
            root.after(0, show_error)

    threading.Thread(target=worker, daemon=True).start()
    root.mainloop()


# --- Singleton lock to prevent multiple instances ---
_singleton_lock_handle = None


def _release_singleton_lock():
    global _singleton_lock_handle
    if _singleton_lock_handle is None:
        return
    try:
        if os.name == 'nt':
            import msvcrt  # noqa: F401
            # msvcrt lock is released on handle close
        else:
            import fcntl
            try:
                fcntl.flock(_singleton_lock_handle.fileno(), fcntl.LOCK_UN)
            except Exception:
                pass
        _singleton_lock_handle.close()
    except Exception:
        pass
    _singleton_lock_handle = None


def try_acquire_singleton(app_id: str = 'LocalReports') -> bool:
    global _singleton_lock_handle
    if _singleton_lock_handle is not None:
        return True
    lock_path = os.path.join(get_base_dir(), f'.{app_id}.lock')
    try:
        fh = open(lock_path, 'w')
        if os.name == 'nt':
            import msvcrt
            try:
                msvcrt.locking(fh.fileno(), msvcrt.LK_NBLCK, 1)
            except OSError:
                fh.close()
                return False
        else:
            import fcntl
            try:
                fcntl.flock(fh.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB)
            except OSError:
                fh.close()
                return False
        _singleton_lock_handle = fh
        atexit.register(_release_singleton_lock)
        return True
    except Exception:
        return False


def notify_already_running():
    try:
        # Prefer native message on Windows to avoid creating full Tk window
        if os.name == 'nt':
            import ctypes
            MB_OK = 0x0
            ctypes.windll.user32.MessageBoxW(0, "Приложение уже запущено.", "Уже запущено", MB_OK)
            return
        # Fallback cross-platform via Tkinter
        import tkinter as tk
        from tkinter import messagebox
        root = tk.Tk()
        root.withdraw()
        messagebox.showinfo("Уже запущено", "Приложение уже запущено.")
        root.destroy()
    except Exception:
        pass


if __name__ == "__main__":
    # CLI flag: -nogui / --nogui → запуск без окна (для вызова из внешних программ)
    no_gui = any(arg in ("-nogui", "--nogui") for arg in sys.argv[1:])

    if not try_acquire_singleton('LocalReports'):
        notify_already_running()
        sys.exit(0)

    if no_gui:
        # headless режим
        main()
    else:
        # окно c индикатором
        run_gui()


